	;*************************************************************************
	; file: kitchen.asm
	;		This is the main application code. We try to minimise changes
	;		to the basic template format, hence we do not place application
	;		code in main.asm
	;*************************************************************************


    ;*************************************************************************
	; include files go here
	;*************************************************************************

    include <P18f27J13.INC>
    include <macros.inc>


	;*************************************************************************
	; extern directives - to gain access to global functions and data from 
	;					  elsewhere
	;*************************************************************************

	extern	LCDCol
	extern	LCDLine
	extern	setPos
	extern	LCDWrite
	extern	bin2ASCIIh
	extern	LCD2ASCII
	extern  usleep
	extern 	LCDOff
	extern 	LCDOn
	extern 	IntOccurred
	extern	LCDCursorOn
	extern	LCDCursorOff

	;*************************************************************************
	; global directives - making variables and functions available elsewhere -
	;					  go here
	;*************************************************************************

	global	kitchenInit
	global	kitchenHandler
	global 	timerAsleep
	global 	timerState
	global  decTimer


	;*************************************************************************
	; Constanst - Symbolic constants and simple macros, defined with the 
	;			  EQU directive, go here.
	;*************************************************************************


	;*************************************************************************
	; Definition of RAM based variables follow the udata directive
	;*************************************************************************

	udata

timerState		res	1	; Set to 1 when editing, 2 when running, 3 when alarming 
slot			res 1	; indicates whether we are incrementing the hours, mins or secs
BCD_H			res	1	; The hours count
BCD_M			res	1	; The minutes count
BCD_S			res	1	; The seconds count


	;*************************************************************************
	; code, such as functions and ROM based data tables, follow the code directive
	;*************************************************************************

	code

txtStartUp:		db	"Kitchen Timer   ",0
txtSetTime:		db	"Set the time    ",0
txtRunning:		db	"Timer Running...",0
txtAlarming:	db	"Timer EXPIRED   ",0


	;*************************************************************************
	; Function: DecTimer
	;			Decrements the hours, minutes & seconds timer.
	;			Carry set if it rolls over 0
	;*************************************************************************

decTimer:
	; if h | m | s == 0, timer has expired. Exit with carry set.
	; If seconds != 0
	;   decrement seconds
	; else 
	;   seconds = 59.
	;   if minutes != 0
	;     decrement minutes
	;   else
	; 	  minutes = 59
	;     decrement hours

	movf	BCD_H, W
	iorwf	BCD_M, W
	iorwf	BCD_S, W
	btfsc	STATUS, Z
	goto	dtExpired

	; If seconds != 0...
	movf	BCD_S, W
	btfsc	STATUS, Z
	goto	dtSecZero
	;   decrement seconds
	call	decTimerS
	goto	dtExit

dtSecZero:
	;   seconds = 59.
	movlw	0x59
	movwf	BCD_S

	;   if minutes != 0...
	movf	BCD_M, W
	btfsc	STATUS, Z
	goto	dtMinZero
	;   decrement minutes
	call	decTimerM
	goto	dtExit
 
dtMinZero:	
	;   minutes = 59.
	movlw	0x59
	movwf	BCD_M
	;     decrement hours	
	call	decTimerH

dtExit:	
	bcf		STATUS, C
	return	
dtExpired:
	bsf		STATUS, C
	return


decTimerH:
	movlw   1 
	subwf   BCD_H 
	addwf   BCD_H, W 
	movlw   0x6 
	btfsc	STATUS, C; 
	movlw  	0x66 
	btfsc	STATUS, DC 
	subwf  	BCD_H
	movf	BCD_H, W
	bcf		STATUS, C
	btfsc	STATUS, Z
	bsf		STATUS, C
	return

decTimerM:
	movlw   1 
	subwf   BCD_M 
	addwf   BCD_M, W 
	movlw   0x6 
	btfsc	STATUS, C; 
	movlw  	0x66 
	btfsc	STATUS, DC 
	subwf  	BCD_M
	movf	BCD_M, W
	bcf		STATUS, C
	btfsc	STATUS, Z
	bsf		STATUS, C
	return

decTimerS:
	movlw   1 
	subwf   BCD_S 
	addwf   BCD_S, W 
	movlw   0x6 
	btfsc	STATUS, C; 
	movlw  	0x66 
	btfsc	STATUS, DC 
	subwf  	BCD_S
	movf	BCD_S, W
	bcf		STATUS, C
	btfsc	STATUS, Z
	bsf		STATUS, C
	return


	;*************************************************************************
	; Function: incTimer
	;			increments the timer time.
	;*************************************************************************

incTimerS:
	movlw	0x67
	addwf	BCD_S
	movlw	0x60
	btfss	STATUS, DC 
	movlw	0x66
	btfss	STATUS, C
	subwf	BCD_S

	movf	BCD_S, W
	sublw	0x60
	btfsc	STATUS, Z
	clrf	BCD_S

	return


incTimerM:
	movlw	0x67
	addwf	BCD_M, F
	movlw	0x60
	btfss	STATUS, DC 
	movlw	0x66
	btfss	STATUS, C
	subwf	BCD_M, F

	movf	BCD_M, W
	sublw	0x60
	btfsc	STATUS, Z
	clrf	BCD_M

	return


incTimerH:
	movlw	0x67
	addwf	BCD_H, F
	movlw	0x60
	btfss	STATUS, DC 
	movlw	0x66
	btfss	STATUS, C
	subwf	BCD_H, F

	movf	BCD_H, W
	sublw	0x60
	btfsc	STATUS, Z
	clrf	BCD_H

	return


	;*************************************************************************
	; Function: kitchenHandler
	;			Called every 2s ( or continuously when the timer is active )
	; 			manages the keypesses and the general "state" of the timer
	;*************************************************************************

kitchenHandler:
	movf	timerState, W
	btfsc	STATUS,Z
	goto	timerASLEEP

	sublw	.1
	btfsc	STATUS,Z
	goto	timerSET

	movf	timerState, W
	sublw	.2
	btfsc	STATUS,Z
	goto	timerRUN

timerALARM:
	; If a key is pressed, turn everything off. 
	call	KeyTimeGetState
	btfss	STATUS, C
	goto	trAlarmCancel
	call	KeyDoGetState
	btfss	STATUS, C
	goto	trAlarmCancel
	return

trAlarmCancel:
	bsf		TRISC, 3	; Turn PWM off.
	call	KeyTimeWaitRelease
	call	KeyDoWaitRelease

	call	LCDOff
	clrf	timerState
	return

timerRUN:
	; If a key is pressed, turn everything off. 
	call	KeyTimeGetState
	btfss	STATUS, C
	goto	trRunCancel
	call	KeyDoGetState
	btfss	STATUS, C
	goto	trRunCancel

	; No button pressed, but things are going on...	
	; If the interrupt occurred, update the display
	movf	IntOccurred, W
	btfsc	STATUS, Z
	return

	sublw	2
	btfss	STATUS, Z
	goto	trUpdate
	
	movlw	3
	movwf	timerState
	call	displayAlarming
	bcf		TRISC, 3	; Turn PWM on
	
trUpdate:
	call	displayTime
	clrf	IntOccurred
	return

trRunCancel:
	call	KeyTimeWaitRelease
	call	KeyDoWaitRelease

	call	LCDOff
	clrf	timerState
	return


timerSET:
	call	KeyTimeGetState
	btfsc	STATUS, C
	goto	ts001

	call	KeyTimeWaitRelease
	movf	slot, W
	btfsc	STATUS, Z
	call	incTimerS

	movf	slot, W
	sublw	0x01
	btfsc	STATUS, Z
	call	incTimerM
	
	movf	slot, W
	sublw	0x02
	btfsc	STATUS, Z
	call	incTimerH

	call	displayTime	
	call	showCursor
	call	KeyTimeWaitRelease
	return

ts001:
	call	KeyDoGetState
	btfsc	STATUS, C
	return
	; wait for key release
	call	KeyDoWaitRelease
	; if slot is currently 2, then start timer
	movf	slot, W
	sublw	0x02
	btfsc	STATUS, Z
	goto	ts002

	incf	slot
	call	showCursor
	return

ts002:
	movlw	2
	movwf	timerState
	call	displayRunning
	call	LCDCursorOff
	; ensure no keys are pressed
	call	KeyDoWaitRelease
	return

timerASLEEP:
	; Any key wakes up LCD, moves to timerSET state

	call	KeyTimeGetState
	btfss	STATUS, C
	goto	taWake

	call	KeyDoGetState
	btfsc	STATUS, C
	
	return

taWake:
	movlw	1
	movwf	timerState
	clrf	slot
	call	kitchenOn

	; Display cursor under seconds
	call	showCursor

	; ensure no keys are pressed
	call	KeyTimeWaitRelease
	call	KeyDoWaitRelease
	return


	;*************************************************************************
	; Function: timerAsleep
	;			returns Z flag set ( and 0 in W ) if timer is in low power mode
	;*************************************************************************

timerAsleep:
	movf	timerState, W
	return


	;*************************************************************************
	; Function: KeyTimeWaitRelease
	;			Waits until the time key is released ( debouncing the pin to be
	;			certain )
	;*************************************************************************

KeyTimeWaitRelease:
	btfss	PORTB, 4
	goto	KeyTimeWaitRelease
	
	; debounce - 20ms
	USLEEP	USLEEP_DEBOUNCE
	btfss	PORTB, 4
	goto	KeyTimeWaitRelease

	return


	;*************************************************************************
	; Function: KeyTimeGetState
	;			Returns the state of the Time key ( debouncing first if it 
	;			is believed to be pressed ) in the carry flag
	; 			Therefore this call may take 1us ( no key ) or 20ms ( key ) 
	;			This is active low : 0 means pressed
	;*************************************************************************

KeyTimeGetState:
	btfss	PORTB, 4
	goto	kttestl

	USLEEP	USLEEP_DEBOUNCE
	btfss	PORTB, 4
	goto	KeyTimeGetState
	
	bsf		STATUS, C

	return

kttestl
	USLEEP	USLEEP_DEBOUNCE
	btfsc	PORTB, 4
	goto	KeyTimeGetState

	bcf		STATUS, C

	return


	;*************************************************************************
	; Function: KeyDoWaitRelease
	;			Waits until the Do key is released ( debouncing the pin to be
	;			certain )
	;*************************************************************************

KeyDoWaitRelease:
	btfss	PORTB, 3
	goto	KeyDoWaitRelease
	
	; debounce - 20ms
	USLEEP	USLEEP_DEBOUNCE
	btfss	PORTB, 3
	goto	KeyDoWaitRelease

	return


	;*************************************************************************
	; Function: KeyDoGetState
	;			Returns the state of the Do key ( debouncing first if it 
	;			is believed to be pressed )
	; 			Therefore this call may take 1us ( no key ) or 20ms ( key ) 
	;			This is active low : 0 means pressed
	;*************************************************************************

KeyDoGetState:
	btfss	PORTB, 3
	goto	kdtestl

	USLEEP	USLEEP_DEBOUNCE
	btfss	PORTB, 3
	goto	KeyDoGetState
	
	bsf		STATUS, C

	return

kdtestl
	USLEEP	USLEEP_DEBOUNCE
	btfsc	PORTB, 3
	goto	KeyDoGetState

	bcf		STATUS, C

	return


	;*************************************************************************
	; Function: displayTime
	;			writes the current setting of the timer to the display
	;*************************************************************************

displayTime:
	clrf	LCDLine
	movlw	.5
	movwf	LCDCol
	call	setPos

	movf	BCD_H, W
	call	bin2ASCIIh
	call	LCD2ASCII

	movlw	.9
	movwf	LCDCol
	call	setPos

	movf	BCD_M, W
	call	bin2ASCIIh
	call	LCD2ASCII

	movlw	.13
	movwf	LCDCol
	call	setPos

	movf	BCD_S, W
	call	bin2ASCIIh
	call	LCD2ASCII

	return


	;*************************************************************************
	; Function: displays a cursor next to the current editable field
	;*************************************************************************

showCursor:
	movf	slot, W
	btfsc	STATUS, Z
	goto	dtCursorS
	sublw	0x01
	btfsc	STATUS, Z
	goto	dtCursorM
	movlw	.7
	goto	dtPutCursor
dtCursorM:
	movlw	.11
	goto	dtPutCursor
dtCursorS:
	movlw	.15
dtPutCursor:
	; Now, display the cursor
	clrf	LCDLine
	movwf	LCDCol	
	call	LCDCursorOn
	return	


	;*************************************************************************
	; Function: displayAlarming
	;			displays the "timer EXPIRED" text
	;*************************************************************************

displayAlarming:
	movlw	LCD_LINE2
	movwf	LCDLine
	clrf	LCDCol
	call	setPos

	movlw	low	txtAlarming
	movwf	TBLPTRL
	movlw	high txtAlarming
	movwf	TBLPTRH
	movlw	upper txtAlarming
	movwf	TBLPTRU
		
da001:
	tblrd*+
	movf	TABLAT, W
	btfsc	STATUS, Z

	return

	call	LCDWrite
	goto	da001


	;*************************************************************************
	; Function: displayRunnning
	;			displays the "timer running" text
	;*************************************************************************

displayRunning:
	movlw	LCD_LINE2
	movwf	LCDLine
	clrf	LCDCol
	call	setPos

	movlw	low	txtRunning
	movwf	TBLPTRL
	movlw	high txtRunning
	movwf	TBLPTRH
	movlw	upper txtRunning
	movwf	TBLPTRU
		
dr001:
	tblrd*+
	movf	TABLAT, W
	btfsc	STATUS, Z

	return

	call	LCDWrite
	goto	dr001


	;*************************************************************************
	; Function: displaySet
	;			displays the "set time" text
	;*************************************************************************

displaySet:
	movlw	LCD_LINE2
	movwf	LCDLine
	clrf	LCDCol
	call	setPos

	movlw	low	txtSetTime
	movwf	TBLPTRL
	movlw	high txtSetTime
	movwf	TBLPTRH
	movlw	upper txtSetTime
	movwf	TBLPTRU
		
ds001:
	tblrd*+
	movf	TABLAT, W
	btfsc	STATUS, Z
	return

	call	LCDWrite
	goto	ds001


	;*************************************************************************
	; Function: kitchenInit
	;			Initialises the kitchen timer application
	;*************************************************************************

kitchenInit:
	; Show startup text
	clrf	LCDLine
	clrf	LCDCol
	call	setPos

	movlw	low	txtStartUp
	movwf	TBLPTRL
	movlw	high txtStartUp
	movwf	TBLPTRH
	movlw	upper txtStartUp
	movwf	TBLPTRU
		
di001:
	tblrd*+
	movf	TABLAT, W
	btfsc	STATUS, Z
	goto	di002

	call	LCDWrite
	goto	di001

di002:
	; A short delay to display the startup banner
	USLEEP	USLEEP_80MS
	USLEEP	USLEEP_80MS
	USLEEP	USLEEP_80MS
	USLEEP	USLEEP_80MS
	USLEEP	USLEEP_80MS

	clrf	timerState		; Start off with the timer running in low power mode
	
	call	LCDOff

	return


	;*************************************************************************
	; Function: kitchenOn
	;			Turns on LCD, resets the timer count.
	;*************************************************************************

kitchenOn:
	clrf	BCD_H
	clrf	BCD_M
	clrf	BCD_S

	call	LCDOn

	clrf	LCDCol
	clrf	LCDLine
	call	setPos
	movlw	'T'
	call	LCDWrite
	movlw	'i'
	call	LCDWrite
	movlw	'm'
	call	LCDWrite
	movlw	'e'
	call	LCDWrite
	
	movlw	' '
	call	LCDWrite
	movlw	'0'
	call	LCDWrite
	movlw	'0'
	call	LCDWrite
	movlw	'h'
	call	LCDWrite

	movlw	' '
	call	LCDWrite
	movlw	'0'
	call	LCDWrite
	movlw	'0'
	call	LCDWrite
	movlw	'm'
	call	LCDWrite

	movlw	' '
	call	LCDWrite
	movlw	'0'
	call	LCDWrite
	movlw	'0'
	call	LCDWrite
	movlw	's'
	call	LCDWrite

	call 	displaySet

	return

	end
